%> @file PlcChar.m
% This file is protected by TIS
% Copyright  2013-2022.

% PlcChar
%>   Represents a character value (in PLC a CHAR) its value defines a single character.
classdef PlcChar < PlcValue
    % ---------- Public properties ----------
    properties (Access = public, Dependent = true)
        %> @brief Gets or sets the value assigned to the PLC value.
        %>   The value assigned to the PLC value.
        Value;
    end % properties

    % ---------- Public constructors ----------
    methods (Access = public)
        %> @brief Initializes a new instance of the PlcChar class using the specified @b address and @b value.
        %>
        %> @param address The PlcAddress of the new PlcChar to which the PLC
        %>                value belongs.
        %> @param value The initial @b value of the new PlcChar.
        %>
        %> @b Errors
        %> * The @b address does not refer to the @b PlcRawType.Byte.
        function this = PlcChar(address, value)
            superArguments{1} = PlcTypeCode.Char;
            superArguments{2} = address;
            superArguments{3} = 0; % Length

            if (nargin == 2)
                superArguments{4} = System.Convert.ToChar(value);
            else
                superArguments{4} = System.Convert.ToChar(' ');
            end

            this = this@PlcValue( ...
                    superArguments{1}, ...
                    superArguments{2}, ...
                    superArguments{3}, ...
                    superArguments{4});
        end
    end % methods

    % ---------- Property accessors ----------
    methods
        %> Gets the value assigned to the PLC value.
        function value = get.Value(this)
            value = char(this.ManagedInstance.GetValue('Value'));
        end

        %> Sets the value assigned to the PLC value.
        function set.Value(this, value)
            this.ManagedInstance.SetValue('Value', System.Convert.ToChar(value));
        end
    end % methods
end

