%> @file PlcDeviceConnection.m
% This file is protected by TIS
% Copyright  2013-2022.

% PlcDeviceConnection
%>   Provides an abstract base class which represents a connection to a device.
classdef (Abstract = true) PlcDeviceConnection < handle
        % ---------- Public constants ----------    
    properties (Access = public, Constant = true)
        %> brief Represents the default timeout value that is used by the PlcDeviceConnection as initial value of all timeout properties.
        DefaultTimeout = 5000;
    end % properties
    
    % ---------- Public properties ----------
    properties (GetAccess = public, SetAccess = private)
        %> @brief Gets the PlcDevice associated with the connection.
        %>   The PlcDevice associated with the connection.
        %>
        %> @b Errors
        %> * The device is not an instance of the PlcDevice class.
        Device;
    end % properties
    
    % ---------- Public properties ----------
    properties (GetAccess = public, Dependent = true)
        %> @brief Gets or sets the time that is used to detect a connection break.
        %>   The time in milliseconds to detect a connection break.
        BreakDetectionTimeout;
        
        %> @brief Gets or sets the wait time before terminating the attempt to establish a connection.
        %>   The time in milliseconds to wait for the connection to connect.
        ConnectTimeout;
        
        %> @brief Gets or sets the wait time before terminating the attempt to receive data.
        %>   The time in milliseconds to wait for the connection to receive.
        ReceiveTimeout;
        
        %> @brief Gets a value that indicates the current state of the connection.
        %>   One of the members defined by the PlcDeviceConnectionState
        %>   enumeration. Which specifies the current state of the
        %>   connection.
        State;
        
        %> @brief Gets or sets the wait time before terminating the attempt to transmit data.
        %>   The time in milliseconds to wait for the connection to
        %>   transmit.
        TransmitTimeout;
        
        %> @brief Gets or sets a value indicating whether a connection break detection is to be used.
        %>   The default value of this property is false.
        UseBreakDetection;
    end % properties
    
    % ---------- Internal properties ----------
    properties (GetAccess = public, SetAccess = protected, Hidden = true)
        %> This property is not intended to be used outside of the toolbox.
        ManagedInstance = 0;
    end % properties
    
    % ---------- Public events ----------
    %> The definition of the events defined by the PlcDeviceConnection.
    events (ListenAccess = public, NotifyAccess = protected)
        %> @brief Occurs when a connection transitions into the closed state.
        Closed;
        
        %> @brief Occurs when a connection transitions into the closing state.
        Closing;

        %> @brief Occurs when a connection transitions into the connected state.
        Connected;

        %> @brief Occurs when a connection transitions into the connecting state.
        Connecting;

        %> @brief Occurs when a connection transitions into the disconnected state.
        Disconnected;

        %> @brief Occurs when a connection transitions into the faulted state.
        Faulted;
        
        %> @brief Occurs when a connection transitions into the opened state.
        Opened;
        
        %> @brief Occurs when a connection transitions into the opening state.
        Opening;
    end % events

    % ---------- Protected constructors ----------
    methods (Access = protected)
        %> @brief Initializes a new instance of the PlcDeviceConnection class using the specified device.
        %>
        %> @param device The PlcDevice associated with the new
        %>               PlcDeviceConnection.
        %>
        %> @b Errors
        %> * The @b device is not an instance of the PlcDevice class.
        function this = PlcDeviceConnection(device)
            SimaticMatLabToolbox.Start();
            this.Device = device;
        end
    end % methods
    
    % ---------- Public destructor ----------
    methods
        %> Performs toolbox-defined tasks associated with freeing, releasing, or
        %> resetting unmanaged resources.
        function delete(this)
            if (~(this.ManagedInstance == 0))
                this.ManagedInstance.Invoke('Close');
            end
        end
    end % methods
    
    % ---------- Property accessors ----------
    methods
        %> Gets the time that is used to detect a connection break.
        function value = get.BreakDetectionTimeout(this)
            value = this.ManagedInstance.GetValue('BreakDetectionTimeout');
        end
        
        %> Sets the time that is used to detect a connection break.
        function set.BreakDetectionTimeout(this, value)
            this.ManagedInstance.SetValue('BreakDetectionTimeout', value);
        end
        
        %> Gets the wait time before terminating the attempt to establish a connection.
        function value = get.ConnectTimeout(this)
            value = this.ManagedInstance.GetValue('ConnectTimeout');
        end
        
        %> Sets the wait time before terminating the attempt to establish a connection.
        function set.ConnectTimeout(this, value)
            this.ManagedInstance.SetValue('ConnectTimeout', value);
        end
        
        %> Gets the PlcDevice associated with the connection.
        function value = get.Device(this)
            value = this.Device;
        end
        
        %> Sets the PlcDevice associated with the connection.
        function set.Device(this, value)
            if (~isa(value, 'PlcDevice'))
                error('The device needs to be an instance of the PlcDevice class.');
            end

            this.Device = value;
        end
        
        %> Gets the wait time before terminating the attempt to receive data.
        function value = get.ReceiveTimeout(this)
            value = this.ManagedInstance.GetValue('ReceiveTimeout');
        end
        
        %> Sets the wait time before terminating the attempt to receive data.
        function set.ReceiveTimeout(this, value)
            this.ManagedInstance.SetValue('ReceiveTimeout', value);
        end
        
        %> Gets a value that indicates the current state of the connection.
        function value = get.State(this)
            value = PlcDeviceConnectionState(int32(this.ManagedInstance.GetValue('State')));
        end
        
        %> Gets the wait time before terminating the attempt to transmit data.
        function value = get.TransmitTimeout(this)
            value = this.ManagedInstance.GetValue('TransmitTimeout');
        end
        
        %> Sets the wait time before terminating the attempt to transmit data.
        function set.TransmitTimeout(this, value)
            this.ManagedInstance.SetValue('TransmitTimeout', value);
        end

        %> Gets a value indicating whether a connection break detection is to be used.
        function value = get.UseBreakDetection(this)
            value = this.ManagedInstance.GetValue('UseBreakDetection');
        end

        %> Sets a value indicating whether a connection break detection is to be used.
        function set.UseBreakDetection(this, value)
            this.ManagedInstance.SetValue('UseBreakDetection', value);
        end
    end % methods

    % ---------- Public methods ----------
    methods (Access = public)
        %> @brief Causes a connection to transition from its current state into the closed state while an established connection to a device will be closed.
        %>
        %> @b Errors
        %> * The connection is in @b PlcDeviceConnectionState.Faulted state and cannot longer be
        %>   closed.
        function Close(this)
            this.ManagedInstance.Invoke('Close');
        end

        %> @brief Causes a connection to transition from the opened state into the connected state while it does fully establish a connection to a device.
        %>
        %> @b Errors
        %> * The connection is not in @b PlcDeviceConnectionState.Opened or
        %>   @b PlcDeviceConnectionState.Disconnected state and is therefore not ready to use or
        %>   the connection is in @b PlcDeviceConnectionState.Faulted state and cannot longer be
        %>   used.
        %>
        %> @b Remarks
        %>
        %> In case there is already a connection fully established (see IsConnected) @b Connect
        %> does not perform any connect attempt.
        function Connect(this)
            this.ManagedInstance.Invoke('Connect');
        end
        
        %> @brief Causes a connection to transition from the created or closed state into the opened state while it does establish a connection to a device.
        %>
        %> @b Errors
        %> * The connection is in @b PlcDeviceConnectionState.Faulted state and cannot longer be
        %>   used.
        function Open(this)
            this.ManagedInstance.Invoke('Open');
        end

        %> @brief Reads the value(s) stored at the specified @b address as a scalar logical value (in PLC a BOOL) or if @b count is specified an array of n logical values (in PLC an ARRAY OF BOOL).
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to read.
        %> @param count The number of values to read from the data area.
        %>
        %> @retval value The value(s) stored at the area specified by @b address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.Bit.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function value = ReadBoolean(this, address, count)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            if (nargin == 2)
                value = this.ManagedInstance.Invoke('ReadBoolean', { address });
            else
                value = zeros(1, count, 'int8');
                managedValue = this.ManagedInstance.Invoke('ReadBoolean', { address, int32(count) });
                
                for index = 1:count
                    value(index) = managedValue(index);
                end
            end
        end
        
        %> @brief Reads the value(s) stored at the specified @b address as a scalar binary value (in PLC a BYTE) or if @b count is specified an array of n binary values (in PLC an ARRAY OF BYTE).
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to read.
        %> @param count The number of values to read from the data area.
        %>
        %> @retval value The value(s) stored at the area specified by @b address.
        %>
        %> @b Errors
        %> * The address is an empty string or does not refer to the
        %>   @b PlcRawType.Byte.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function value = ReadByte(this, address, count)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
                        
            if (nargin == 2)
                value = this.ManagedInstance.Invoke('ReadByte', { address });
            else
                value = zeros(1, count, 'uint8');
                managedValue = this.ManagedInstance.Invoke('ReadByte', { address, int32(count) });
                
                for index = 1:count
                    value(index) = managedValue(index);
                end
            end
        end
        
        %> @brief Reads the value(s) stored at the specified @b address as a scalar character value (in PLC a CHAR) or if @b count is specified an array of n character values (in PLC an ARRAY OF CHAR).
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to read.
        %> @param count The number of values to read from the data area.
        %>
        %> @retval value The value(s) stored at the area specified by @b address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.Byte.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function value = ReadChar(this, address, count)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            if (nargin == 2)
                value = this.ManagedInstance.Invoke('ReadChar', { address });
            else
                % disp('WARNING: There is an internal error reading n values.');
                
                value = char.empty(count, 0);
                managedValue = this.ManagedInstance.Invoke('ReadChar', { address, int32(count) });
                
                for index = 1:count
                    value(index) = managedValue(index);
                end
            end
        end

        %> @brief Reads the value stored at the specified @b address as a scalar date number (in PLC a DATE).
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to read.
        %>
        %> @retval value The value stored at the area specified by @b address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.Word.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function value = ReadDate(this, address)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            managedValue = this.ManagedInstance.Invoke('ReadDate', { address });
            value = datenum( ...
                    double(managedValue.Year), ...
                    double(managedValue.Month), ...
                    double(managedValue.Day));
        end
        
        %> @brief Reads the value stored at the specified @b address as a scalar date number (in PLC a DATE_AND_TIME).
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to read.
        %>
        %> @retval value The value stored at the area specified by @b address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.Byte.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function value = ReadDateTime(this, address)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            managedValue = this.ManagedInstance.Invoke('ReadDateTime', { address });
            value = datenum( ...
                    double(managedValue.Year), ...
                    double(managedValue.Month), ...
                    double(managedValue.Day), ...
                    double(managedValue.Hour), ...
                    double(managedValue.Minute), ...
                    double(managedValue.Second));
        end

        %> @brief Reads the value(s) stored at the specified @b address as a scalar double value (in PLC a DOUBLE) or if @b count is specified an array of n double values (in PLC an ARRAY OF DOUBLE).
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to read.
        %> @param count The number of values to read from the data area.
        %>
        %> @retval value The value(s) stored at the area specified by @b address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.DWord.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function value = ReadDouble(this, address, count)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            if (nargin == 2)
                value = this.ManagedInstance.Invoke('ReadDouble', { address });
            else                
                value = zeros(1, count, 'double');
                managedValue = this.ManagedInstance.Invoke('ReadDouble', { address, int32(count) });
                
                for index = 1:count
                    value(index) = managedValue(index);
                end
            end
        end
        
        %> @brief Reads the value(s) stored at the specified @b address as a scalar int16 value (in PLC a INT) or if @b count is specified an array of n int16 values (in PLC an ARRAY OF INT).
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to read.
        %> @param count The number of values to read from the data area.
        %>
        %> @retval value The value(s) stored at the area specified by @b address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.Word.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function value = ReadInt16(this, address, count)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            if (nargin == 2)
                value = this.ManagedInstance.Invoke('ReadInt16', { address });
            else                
                value = zeros(1, count, 'int16');
                managedValue = this.ManagedInstance.Invoke('ReadInt16', { address, int32(count) });
                
                for index = 1:count
                    value(index) = managedValue(index);
                end
            end
        end
        
        %> @brief Reads the value(s) stored at the specified @b address as a scalar int32 value (in PLC a DINT) or if @b count is specified an array of n int32 values (in PLC an ARRAY OF DINT).
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to read.
        %> @param count The number of values to read from the data area.
        %>
        %> @retval value The value(s) stored at the area specified by @b address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.DWord.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function value = ReadInt32(this, address, count)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            if (nargin == 2)
                value = this.ManagedInstance.Invoke('ReadInt32', { address });
            else                
                value = zeros(1, count, 'int32');
                managedValue = this.ManagedInstance.Invoke('ReadInt32', { address, int32(count) });
                
                for index = 1:count
                    value(index) = managedValue(index);
                end
            end
        end
        
        %> @brief Reads the value(s) stored at the specified @b address as a scalar single value (in PLC a REAL) or if @b count is specified an array of n single values (in PLC an ARRAY OF REAL).
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to read.
        %> @param count The number of values to read from the data area.
        %>
        %> @retval value The value(s) stored at the area specified by @b address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.DWord.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function value = ReadReal(this, address, count)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            if (nargin == 2)
                value = this.ManagedInstance.Invoke('ReadReal', { address });
            else                
                value = zeros(1, count, 'single');
                managedValue = this.ManagedInstance.Invoke('ReadReal', { address, int32(count) });
                
                for index = 1:count
                    value(index) = managedValue(index);
                end
            end
        end

        %> @brief Reads the value stored at the specified @b address as a scalar date number value (in PLC a S5TIME), while the date number does only express time information.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to read.
        %>
        %> @retval value The value stored at the area specified by @b address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.Word.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function value = ReadS5Time(this, address)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            managedValue = this.ManagedInstance.Invoke('ReadS5Time', { address });
            value = datenum( ...
                    double(0), ...
                    double(0), ...
                    double(0), ...
                    double(managedValue.Hours), ...
                    double(managedValue.Minutes), ...
                    double(managedValue.Seconds));
        end

        %> @brief Reads the value stored at the specified @b address as a scalar string value (in PLC a TEXT).
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to read.
        %> @param length The length of the string value to read.
        %>
        %> @retval value The value stored at the area specified by @b address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.Byte.
        %> * The @b length is out of the bounds defined by @b PlcString.MinLength
        %>   and @b PlcString.MaxLength.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function value = ReadString(this, address, length)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            if (nargin == 2)
                value = char(this.ManagedInstance.Invoke('ReadString', { address }));
            else
                value = char(this.ManagedInstance.Invoke('ReadString', { address, int32(length) }));
            end
        end

        %> @brief Reads the value stored at the specified @b address as a scalar date number value (in PLC a TIME), while the date number does only express time information.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to read.
        %>
        %> @retval value The value stored at the area specified by @b address.
        %>
        %> @b Errors
        %> * The address is an empty string or does not refer to the
        %>   @b PlcRawType.DWord.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function value = ReadTime(this, address)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            managedValue = this.ManagedInstance.Invoke('ReadTime', { address });
            value = datenum( ...
                    double(0), ...
                    double(0), ...
                    double(0), ...
                    double(managedValue.Hours), ...
                    double(managedValue.Minutes), ...
                    double(managedValue.Seconds));
        end
        
        %> @brief Reads the value stored at the specified @b address as a scalar date number value (in PLC a TIME_OF_DAY), while the date number does only express time information.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to read.
        %>
        %> @retval value The value stored at the area specified by @b address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.DWord.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function value = ReadTimeOfDay(this, address)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            managedValue = this.ManagedInstance.Invoke('ReadTimeOfDay', { address });
            value = datenum( ...
                    double(0), ...
                    double(0), ...
                    double(0), ...
                    double(managedValue.Hours), ...
                    double(managedValue.Minutes), ...
                    double(managedValue.Seconds));
        end
        
        %> @brief Reads the value(s) stored at the specified @b address as a scalar uint16 value (in PLC a WORD) or if @b count is specified an array of n uint16 values (in PLC an ARRAY OF WORD).
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to read.
        %> @param count The number of values to read from the data area.
        %>
        %> @retval value The value(s) stored at the area specified by @b address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.Word.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function value = ReadUInt16(this, address, count)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            if (nargin == 2)
                value = this.ManagedInstance.Invoke('ReadUInt16', { address });
            else                
                value = zeros(1, count, 'uint16');
                managedValue = this.ManagedInstance.Invoke('ReadUInt16', { address, int32(count) });
                
                for index = 1:count
                    value(index) = managedValue(index);
                end
            end
        end
        
        %> @brief Reads the value(s) stored at the specified @b address as a scalar uint32 value (in PLC a DWORD) or if @b count is specified an array of n uint32 values (in PLC an ARRAY OF DWORD).
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to read.
        %> @param count The number of values to read from the data area.
        %>
        %> @retval value The value(s) stored at the area specified by @b address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.DWord.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function value = ReadUInt32(this, address, count)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            if (nargin == 2)
                value = this.ManagedInstance.Invoke('ReadUInt32', { address });
            else                
                value = zeros(1, count, 'uint32');
                managedValue = this.ManagedInstance.Invoke('ReadUInt32', { address, int32(count) });
                
                for index = 1:count
                    value(index) = managedValue(index);
                end
            end
        end
        
        %> @brief Reads the values specified.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param values The values to be read (as cell array).
        %>
        %> @retval value An array of the values read.
        %>
        %> @b Errors
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function value = ReadValues(this, values)
            managedValues = IPS7Lnk.Advanced.MatLab.Factory.CreateValueArray(length(values));
            
            if (iscell(values))
                for index = 1:length(values)
                    managedValues.Set(index - 1, values{index}.ManagedInstance);
                end
            else
                managedValues.Set(0, values.ManagedInstance);
            end

            results = cell(1, length(values));
            this.ManagedInstance.Invoke('ReadValues', { managedValues });

            if (iscell(values))
                for index = 1:length(values)
                    results{index} = values{index}.Value;
                end
            else
                results{1} = values.Value;
            end

            value = results;
        end









        %> @brief Writes the logical value(s) (in PLC BOOL or ARRAY OF BOOL) to the data area specified by @b address.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to that the values are to
        %>                be written.
        %> @param values The logical values to write to the area specified by
        %>               address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.Bit.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function WriteBoolean(this, address, values)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            managedValues = NET.createArray( ...
                    'System.Boolean', length(values));
                
            if (iscell(values))
                % In case there values is a cell array.
                for index = 1:length(values)
                    managedValues.Set(index - 1, System.Convert.ToBoolean(values{index}));
                end
            elseif (length(values) > 1)
                % In case there values is an array.
                for index = 1:length(values)
                    managedValues.Set(index - 1, System.Convert.ToBoolean((values(index))));
                end
            else
                managedValues.Set(0, ~(values == 0));    
            end

            this.ManagedInstance.Invoke('WriteBoolean', { address, managedValues });
        end
        
        %> @brief Writes the binary value(s) (in PLC BYTE or ARRAY OF BYTE) to the data area specified by @b address.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to that the values are to
        %>                be written.
        %> @param values The binary values to write to the area specified by
        %>               address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.Byte.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function WriteByte(this, address, values)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            managedValues = NET.createArray( ...
                    'System.Byte', length(values));

            if (iscell(values))
                % In case there values is a cell array.
                for index = 1:length(values)
                    managedValues.Set(index - 1, values{index});
                end
            elseif (length(values) > 1)
                % In case there values is an array.
                for index = 1:length(values)
                    managedValues.Set(index - 1, values(index));
                end
            else
                managedValues.Set(0, values);
            end
            
            this.ManagedInstance.Invoke('WriteByte', { address, managedValues });
        end
        
        %> @brief Writes the character value(s) (in PLC CHAR or ARRAY OF CHAR) to the data area specified by @b address.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to that the values are to
        %>                be written.
        %> @param values The character values to write to the area specified by
        %>               address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.Byte.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function WriteChar(this, address, values)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            managedValues = NET.createArray( ...
                    'System.Char', length(values));

            if (iscell(values))
                % In case there values is a cell array.
                for index = 1:length(values)
                    managedValues.Set(index - 1, values{index});
                end
            elseif (length(values) > 1)
                % In case there values is an array.
                for index = 1:length(values)
                    managedValues.Set(index - 1, values(index));
                end
            else
                managedValues.Set(0, values);
            end
            
            this.ManagedInstance.Invoke('WriteChar', { address, managedValues });
        end

        %> @brief Writes the date number (in PLC DATE) to the data area specified by @b address.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to that the values are to
        %>                be written.
        %> @param value The date number to write to the area specified by
        %>              address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.Word.
        %> * The value is out of the bounds defined by @b PlcDate.MinValue
        %>   and @b PlcDate.MaxValue.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function WriteDate(this, address, value)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            this.ManagedInstance.Invoke( ...
                    'WriteDate', ...
                    { address, System.DateTime.Parse(datestr(value)) });
        end
        
        %> @brief Writes the date number (in PLC DATE_AND_TIME) to the data area specified by @b address.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to that the values are to
        %>                be written.
        %> @param value The date number to write to the area specified by
        %>              address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.Byte.
        %> * The value is out of the bounds defined by @b PlcDateTime.MinValue
        %>   and @b PlcDateTime.MaxValue.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function WriteDateTime(this, address, value)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            this.ManagedInstance.Invoke( ...
                    'WriteDateTime', ...
                    { address, System.DateTime.Parse(datestr(value)) });
        end

        %> @brief Writes the double value(s) (in PLC DOUBLE or ARRAY OF DOUBLE) to the data area specified by @b address.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to that the values are to
        %>                be written.
        %> @param values The double values to write to the area specified by
        %>                address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.DWord.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function WriteDouble(this, address, values)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            managedValues = NET.createArray( ...
                    'System.Double', length(values));

            if (iscell(values))
                % In case there values is a cell array.
                for index = 1:length(values)
                    managedValues.Set(index - 1, values{index});
                end
            elseif (length(values) > 1)
                % In case there values is an array.
                for index = 1:length(values)
                    managedValues.Set(index - 1, values(index));
                end
            else
                managedValues.Set(0, values);
            end
            
            this.ManagedInstance.Invoke('WriteDouble', { address, managedValues });
        end
        
        %> @brief Writes the int16 value(s) (in PLC INT or ARRAY OF INT) to the data area specified by @b address.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to that the values are to
        %>                be written.
        %> @param values The int16 values to write to the area specified by
        %>               address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.Word.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function WriteInt16(this, address, values)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            managedValues = NET.createArray( ...
                    'System.Int16', length(values));

            if (iscell(values))
                % In case there values is a cell array.
                for index = 1:length(values)
                    managedValues.Set(index - 1, values{index});
                end
            elseif (length(values) > 1)
                % In case there values is an array.
                for index = 1:length(values)
                    managedValues.Set(index - 1, values(index));
                end
            else
                managedValues.Set(0, values);
            end
            
            this.ManagedInstance.Invoke('WriteInt16', { address, managedValues });
        end
        
        %> @brief Writes the int32 value(s) (in PLC DINT or ARRAY OF DINT) to the data area specified by @b address.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to that the values are to
        %>                be written.
        %> @param values The int32 values to write to the area specified by
        %>               address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.DWord.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function WriteInt32(this, address, values)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            managedValues = NET.createArray( ...
                    'System.Int32', length(values));

            if (iscell(values))
                % In case there values is a cell array.
                for index = 1:length(values)
                    managedValues.Set(index - 1, values{index});
                end
            elseif (length(values) > 1)
                % In case there values is an array.
                for index = 1:length(values)
                    managedValues.Set(index - 1, values(index));
                end
            else
                managedValues.Set(0, values);
            end
            
            this.ManagedInstance.Invoke('WriteInt32', { address, managedValues });
        end
        
        %> @brief Writes the single value(s) (in PLC REAL or ARRAY OF REAL) to the data area specified by @b address.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to that the values are to
        %>                be written.
        %> @param values The single values to write to the area specified by
        %>                address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.DWord.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function WriteReal(this, address, values)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            managedValues = NET.createArray( ...
                    'System.Single', length(values));

            if (iscell(values))
                % In case there values is a cell array.
                for index = 1:length(values)
                    managedValues.Set(index - 1, values{index});
                end
            elseif (length(values) > 1)
                % In case there values is an array.
                for index = 1:length(values)
                    managedValues.Set(index - 1, values(index));
                end
            else
                managedValues.Set(0, values);
            end
            
            this.ManagedInstance.Invoke('WriteReal', { address, managedValues });
        end

        %> @brief Writes the time part of the date number (in PLC S5TIME) to the data area specified by @b address.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to that the values are to
        %>                be written.
        %> @param value The date number to write to the area specified by
        %>              address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.Word.
        %> * The value is out of the bounds defined by @b PlcS5Time.MinValue
        %>   and @b PlcS5Time.MaxValue.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function WriteS5Time(this, address, value)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            this.ManagedInstance.Invoke( ...
                    'WriteS5Time', ...
                    { address, System.DateTime.Parse(datestr(value)).TimeOfDay });
        end

        %> @brief Writes the string (in PLC TEXT) to the data area specified by @b address.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to that the values are to
        %>                be written.
        %> @param value The string to write to the area specified by address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.Byte.
        %> * The length of value is out of the bounds defined by
        %>   @b PlcString.MinLength and @b PlcString.MaxLength.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function WriteString(this, address, value)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            this.ManagedInstance.Invoke('WriteString', { address, value });
        end

        %> @brief Writes the time part of the date number (in PLC TIME) to the data area specified by @b address.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to that the values are to
        %>                be written.
        %> @param value The date number to write to the area specified by
        %>               address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.DWord.
        %> * The value is out of the bounds defined by @b PlcTime.MinValue
        %>   and @b PlcTime.MaxValue.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function WriteTime(this, address, value)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            this.ManagedInstance.Invoke( ...
                    'WriteTime', ...
                    { address, System.DateTime.Parse(datestr(value)).TimeOfDay });
        end
        
        %> @brief Writes the time part of the date number (in PLC TIME_OF_DAY) to the data area specified by @b address.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to that the values are to
        %>                be written.
        %> @param value The date number to write to the area specified by
        %>              address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.DWord.
        %> * The value is out of the bounds defined by @b PlcTimeOfDay.MinValue
        %>   and @b PlcTimeOfDay.MaxValue.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function WriteTimeOfDay(this, address, value)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            this.ManagedInstance.Invoke( ...
                    'WriteTimeOfDay', ...
                    { address, System.DateTime.Parse(datestr(value)).TimeOfDay });
        end
        
        %> @brief Writes the uint16 value(s) (in PLC WORD or ARRAY OF WORD) to the data area specified by @b address.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to that the values are to
        %>                be written.
        %> @param values The uint16 values to write to the area specified by
        %>               address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.Word.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function WriteUInt16(this, address, values)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            managedValues = NET.createArray( ...
                    'System.UInt16', length(values));

            if (iscell(values))
                % In case there values is a cell array.
                for index = 1:length(values)
                    managedValues.Set(index - 1, values{index});
                end
            elseif (length(values) > 1)
                % In case there values is an array.
                for index = 1:length(values)
                    managedValues.Set(index - 1, values(index));
                end
            else
                managedValues.Set(0, values);
            end
            
            this.ManagedInstance.Invoke('WriteUInt16', { address, managedValues });
        end
        
        %> @brief Writes the uint32 value(s) (in PLC DWORD or ARRAY OF DWORD) to the data area specified by @b address.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param address The address of the data area to that the values are to
        %>                be written.
        %> @param values The uint32 values to write to the area specified by
        %>               address.
        %>
        %> @b Errors
        %> * The @b address is an empty string or does not refer to the
        %>   @b PlcRawType.DWord.
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function WriteUInt32(this, address, values)
            if (isa(address, 'PlcAddress'))
                address = address.ManagedInstance;
            end
            
            managedValues = NET.createArray( ...
                    'System.UInt32', length(values));

            if (iscell(values))
                % In case there values is a cell array.
                for index = 1:length(values)
                    managedValues.Set(index - 1, values{index});
                end
            elseif (length(values) > 1)
                % In case there values is an array.
                for index = 1:length(values)
                    managedValues.Set(index - 1, values(index));
                end
            else
                managedValues.Set(0, values);
            end
            
            this.ManagedInstance.Invoke('WriteUInt32', { address, managedValues });
        end
        
        %> @brief Writes the values specified.
        %>
        %> @param this The instance of the PlcDeviceConnection class to operate on.
        %> @param values The values to write (as cell array).
        %>
        %> @b Errors
        %> * The connection is not in @b PlcDeviceConnectionState.Opened,
        %>   @b PlcDeviceConnectionState.Connected or @b PlcDeviceConnectionState.Disconnected
        %>   state and is therefore not ready to use or the connection is in
        %>   @b PlcDeviceConnectionState.Faulted state and cannot longer be used.
        function WriteValues(this, values)
            managedValues = IPS7Lnk.Advanced.MatLab.Factory.CreateValueArray(length(values));

            if (iscell(values))
                for index = 1:length(values)
                    managedValues.Set(index - 1, values{index}.ManagedInstance);
                end
            else
                managedValues.Set(0, values.ManagedInstance);
            end

            this.ManagedInstance.Invoke('WriteValues', { managedValues });
        end
    end % methods
end

