%> @file PlcTime.m
% This file is protected by TIS
% Copyright  2013-2022.

% PlcTime
%>   Represents a date number value, where only the time of day is expressed, (in PLC a TIME) its value defines an time interval.
classdef PlcTime < PlcValue
    % ---------- Public properties ----------
    properties (Access = public, Dependent = true)
        %> @brief Gets or sets the value assigned to the PLC value.
        %>   The value assigned to the PLC value.
        Value;
    end % properties

    % ---------- Public constructors ----------
    methods (Access = public)
        %> @brief Initializes a new instance of the PlcTime class using the specified @b address and @b value.
        %>
        %> @param address The PlcAddress of the new PlcTime to which the PLC
        %>                value belongs.
        %> @param value The initial value of the new PlcTime.
        %>
        %> @b Errors
        %> * The @b address does not refer to the @b PlcRawType.DWord.
        function this = PlcTime(address, value)
            superArguments{1} = PlcTypeCode.Time;
            superArguments{2} = address;
            superArguments{3} = 0; % Length

            if (nargin == 2)
                superArguments{4} = System.DateTime.Parse(datestr(value)).TimeOfDay;
            else
                superArguments{4} = System.DateTime.Now.TimeOfDay;
            end

            this = this@PlcValue( ...
                    superArguments{1}, ...
                    superArguments{2}, ...
                    superArguments{3}, ...
                    superArguments{4});
        end
    end % methods

    % ---------- Property accessors ----------
    methods
        %> Gets the value assigned to the PLC value.
        function value = get.Value(this)
            managedValue = this.ManagedInstance.GetValue('Value');
            
            value = datenum( ...
                    double(0), ...
                    double(0), ...
                    double(0), ...
                    double(managedValue.Hours), ...
                    double(managedValue.Minutes), ...
                    double(managedValue.Seconds));
        end

        %> Sets the value assigned to the PLC value.
        function set.Value(this, value)
            this.ManagedInstance.SetValue('Value', System.DateTime.Parse(datestr(value)).TimeOfDay);
        end
    end % methods
end

