%> @file SimaticDevice.m
% This file is protected by TIS
% Copyright  2013-2022.

% SimaticDevice
%>   Represents a single Siemens AG specific programmable logic controller (PLC).
classdef SimaticDevice < PlcDevice
    % ---------- Public properties ----------
    properties (Access = public, Dependent = true)
        %> @brief Gets or sets a value that indicates how the connection will operate on the device.
        %>   One of the members defined by the SimaticChannelType enumeration.
        %>
        %> @b Errors
        %> * The type is not an instance of the SimaticChannelType enumertion
        %>   class.
        ChannelType;

        %> @brief Gets or sets a value indicating the specific type of device represented.
        %>   One of the members defined by the SimaticDeviceType
        %>   enumeration. Which specifies the concrete type of Siemens
        %>   device to represent.
        %>
        %> @b Errors
        %> * The type is not an instance of the SimaticDeviceType enumertion
        %>   class.
        Type;
    end % properties

    % ---------- Public constructors ----------
    methods (Access = public)
        %> @brief Initializes a new instance of the SimaticDevice class using the specified @b endPoint and @b type.
        %>
        %> @param endPoint The PlcDeviceEndPoint which identifies the network
        %>                  address of the device.
        %> @param type The type of Siemens device to represent.
        %>
        %> @b Errors
        %> * The @b type is not an instance of the SimaticDeviceType
        %>   enumeration class.
        function this = SimaticDevice(endPoint, type)
            SimaticMatLabToolbox.Start();

            this = this@PlcDevice(endPoint);
            deviceType = SimaticDeviceType.S7300_400;

            if (nargin == 2)
                deviceType = type;
            end

            this.ManagedInstance = IPS7Lnk.Advanced.MatLab.Factory.CreateDevice( ...
                    endPoint.ManagedInstance, int32(deviceType));
        end
    end % methods

    % ---------- Property accessors ----------
    methods
        %> Gets a value that indicates how the connection will operate on the device.
        function value = get.ChannelType(this)
            value = SimaticChannelType(int32(this.ManagedInstance.GetValue('ChannelType')));
        end

        %> Sets a value that indicates how the connection will operate on the device.
        function set.ChannelType(this, value)
            if (~isa(value, 'SimaticChannelType'))
                error('The type needs to be an instance of the SimaticChannelType enumeration class.');
            end

            IPS7Lnk.Advanced.MatLab.Helper.ApplyChannelType(this.ManagedInstance, int32(value));
        end

        %> Gets a value indicating the specific type of device represented.
        function value = get.Type(this)
            value = SimaticDeviceType(int32(this.ManagedInstance.GetValue('Type')));
        end

        %> Sets a value indicating the specific type of device represented.
        function set.Type(this, value)
            if (~isa(value, 'SimaticDeviceType'))
                error('The type needs to be an instance of the SimaticDeviceType enumeration class.');
            end

            IPS7Lnk.Advanced.MatLab.Helper.ApplyDeviceType(this.ManagedInstance, int32(value));
        end
    end % methods

    % ---------- Public methods ----------
    methods (Access = public)
        %> @brief Creates a new instance of an object implementing the PlcDeviceConnection class which is associated with the device, if <paramref name="createNew"/> is equals to the value true.
        %>
        %> @param createNew A value indicating whether a new instance is to be created (the value
        %>        true) or if there is already an existing instance of the PlcDeviceConnection
        %>        class in usable state is to be returned (the value false).
        %>
        %> @retval An instance of the PlcDeviceConnection class which is associated with the device.
        %>
        %> @b Remarks
        %>
        %> Any changes made on @b EndPoint and any changes made on the configuration of the
        %> returned @b PlcDeviceConnection will take affect when calling
        %> @b PlcDeviceConnection.Open. Any further changes made on the @b EndPoint or on the
        %> @b PlcDeviceConnection> instance will when only take affect after closing (using
        %> @b PlcDeviceConnection.Close) and re-opening (using PlcDeviceConnection.Open) the
        %> connection.
        %>
        %> In multi-threaded environments it is important to take care about which thread does
        %> open/close a connection and which threads do use the same connection concurrent.
        %> Scenarios there multiple threads do share the same connection instance have to ensure
        %> that changes to the @b PlcDevice (e.g. @b EndPoint changes) to that a
        %> @b PlcDeviceConnection does belong or any changes made on the connection itself will
        %> take affect in all threads (which share the same connection instance) immediately after
        %> closing and re-opening the connection.
        %>
        %> If there was a connection created for the first time using this @b PlcDevice instance
        %> these connection will be always returned by any subsequent calls to
        %> @b CreateConnection(bool) using @b createNew with the value false. This is the case
        %> until the cached connection will be discarded. A cached connection will be discarded
        %> then it is not in created nor in closed state and any changes are made to the
        %> configuration of this PlcDevice (e.g. @b EndPoint).
        function connection = CreateConnection(this, createNew)
            isNewRequired = nargin == 1 ...
                    || createNew ...
                    || (this.Connection == 0 ...
                    || this.Connection.State == PlcDeviceConnectionState.Faulted);

            if (isNewRequired)
                this.Connection = SimaticDeviceConnection(this);
            end

            connection = this.Connection;
        end
    end % methods
end

