%> @file IPDeviceEndPoint.m
% This file is protected by TIS
% Copyright  2013-2022.

% IPDeviceEndPoint
%>   Represents a network endpoint as an IP address, rack and slot number of a PLC device.
classdef IPDeviceEndPoint < handle
    % ---------- Public constants ----------    
    properties (Access = public, Constant = true)
        %> @brief Specifies the default value that is by default assigned to the @b Rack property.
        DefaultRack = 0;

        %> @brief Specifies the default value that is by default assigned to the @b Slot property.
        DefaultSlot = 2;

        %> @brief Specifies the maximum value that can be assigned to the @b Rack property.
        MaxRack = 65535;

        %> @brief Specifies the maximum value that can be assigned to the @b Slot property.
        MaxSlot = 65535;

        %> @brief Specifies the minimum value that can be assigned to the @b Rack property.
        MinRack = 0;

        %> @brief Specifies the minimum value that can be assigned to the @b Slot property.
        MinSlot = 0;
    end % properties

    % ---------- Public properties ----------
    properties (Access = public, Dependent = true)
        %> @brief Gets or sets the IP address of the endpoint.
        %>   The IP address of the endpoint.
        %>
        %> @b Errors
        %> * The address is not a valid IP address.
        Address;

        %> @brief Gets or sets the rack number of the endpoint.
        %>   A numeric value in the range @b MinRack to @b MaxRack indicating
        %>   the rack number of the endpoint.
        %>
        %> @b Errors
        %> * The rack is out of the bounds defined by @b MinRack or @b MaxRack.
        Rack;

        %> @brief Gets or sets the slot number of the endpoint.
        %>   A numeric value in the range @b MinSlot to @b MaxSlot indicating
        %>   the slot number of the endpoint.
        %>
        %> @b Errors
        %> * The slot is out of the bounds defined by @b MinSlot or @b MaxSlot.
        Slot;
    end % properties

    properties (GetAccess = public, Dependent = true)
        %> @brief Gets a value indicating whether a custom rack has been associated with this endpoint.
        %>   The value true, if there was a custom rack assigned to this endpoint; otherwise the
        %>   value false.
        %>
        %> @b Remarks
        %>
        %> In case there is no custom rack number defined the PLC driver does decide on its own
        %> which rack is to be used by default to address the PLC device.
        IsCustomRack;

        %> @brief Gets a value indicating whether a custom slot has been associated with this endpoint.
        %>   The value true, if there was a custom slot assigned to this endpoint; otherwise the
        %>   value false.
        %>
        %> @b Remarks
        %>
        %> In case there is no custom slot number defined the PLC driver does decide on its own
        %> which slot is to be used by default to address the PLC device.
        IsCustomSlot;
    end % properties
    
    % ---------- Internal properties ----------
    properties (GetAccess = public, SetAccess = private, Hidden = true)
        ManagedInstance = 0;
    end % properties
    
    % ---------- Public constructors ----------
    methods (Access = public)
        %> @brief Initializes a new instance of the IPDeviceEndPoint class with the specified @b address, @b rack number and @b slot number.
        %>
        %> @param address The IP address of the endpoint.
        %> @param rack The rack number associated with the address.
        %> @param slot The slot number associated with the address.
        %> 
        %> @b Errors
        %> * The @b address is not a valid IP address.
        %> * The @b rack is out of the bounds defined by @b MinRack or @b MaxRack.
        %> * The @b slot is out of the bounds defined by @b MinSlot or @b MaxSlot.
        function this = IPDeviceEndPoint(address, rack, slot)
            SimaticMatLabToolbox.Start();

            this.ManagedInstance = IPS7Lnk.Advanced.MatLab.Factory.CreateEndPoint('0.0.0.0');

            if (nargin == 0)
                this.Address = '0.0.0.0';
                this.Rack = 0;
                this.Slot = 2;
            else
                if (nargin >= 1)
                    this.Address = address;
                end

                if (nargin >= 2)
                    this.Rack = rack;
                end

                if (nargin == 3)
                    this.Slot = slot;
                end
            end
        end
    end % methods

    % ---------- Property accessors ----------
    methods
        %> Gets the IP address of the endpoint.
        function value = get.Address(this)
            value = char(this.ManagedInstance.GetValue('Address').ToString());
        end

        %> Sets the IP address of the endpoint.
        function set.Address(this, value)
            this.ManagedInstance.SetValue('Address', System.Net.IPAddress.Parse(value));
        end

        %> Gets a value indicating whether a custom rack has been associated with this endpoint.
        function value = get.IsCustomRack(this)
            value = this.ManagedInstance.GetValue('IsCustomRack');
        end

        %> Gets a value indicating whether a custom slot has been associated with this endpoint.
        function value = get.IsCustomSlot(this)
            value = this.ManagedInstance.GetValue('IsCustomSlot');
        end

        %> Gets the rack number of the endpoint.
        function value = get.Rack(this)
            value = this.ManagedInstance.GetValue('Rack');
        end

        %> Sets the rack number of the endpoint.
        function set.Rack(this, value)
            if (value < 0 || value > 65535)
                error('The rack is out of the bounds specified by IPDeviceEndPoint.MinRack and IPDeviceEndPoint.MaxRack.');
            end
            
            this.ManagedInstance.SetValue('Rack', value);
        end

        %> Gets the slot number of the endpoint.
        function value = get.Slot(this)
            value = this.ManagedInstance.GetValue('Slot');
        end

        %> Sets the slot number of the endpoint.
        function set.Slot(this, value)
            if (value < 0 || value > 65535)
                error('The slot is out of the bounds specified by IPDeviceEndPoint.MinSlot and IPDeviceEndPoint.MaxSlot.');
            end

            this.ManagedInstance.SetValue('Slot', value);
        end
    end % methods
end

