%> @file PlcByte.m
% This file is protected by TIS
% Copyright  2013-2022.

% PlcByte
%>   Represents a binary value (in PLC a BYTE) its value defines a 8-bit unsigned integer value.
classdef PlcByte < PlcValue
    % ---------- Public properties ----------
    properties (Access = public, Dependent = true)
        %> @brief Gets or sets the value assigned to the PLC value.
        %>   The value assigned to the PLC value.
        Value;
    end % properties

    % ---------- Public constructors ----------
    methods (Access = public)
        %> @brief Initializes a new instance of the PlcByte class using the specified @b address and @b value.
        %>
        %> @param address The PlcAddress of the new PlcByte to which the PLC
        %>                value belongs.
        %> @param value The initial @b value of the new PlcByte.
        %>
        %> @b Errors
        %> * The @b address does not refer to the @b PlcRawType.Byte.
        function this = PlcByte(address, value)
            superArguments{1} = PlcTypeCode.Byte;
            superArguments{2} = address;
            superArguments{3} = 0; % Length

            if (nargin == 2)
                superArguments{4} = System.Convert.ToByte(value);
            else
                superArguments{4} = System.Convert.ToByte(0);
            end

            this = this@PlcValue( ...
                    superArguments{1}, ...
                    superArguments{2}, ...
                    superArguments{3}, ...
                    superArguments{4});
        end
    end % methods

    % ---------- Property accessors ----------
    methods
        %> Gets the value assigned to the PLC value.
        function value = get.Value(this)
            value = this.ManagedInstance.GetValue('Value');
        end

        %> Sets the value assigned to the PLC value.
        function set.Value(this, value)
            this.ManagedInstance.SetValue('Value', System.Convert.ToByte(value));
        end
    end % methods
end

