%> @file PlcByteArray.m
% This file is protected by TIS
% Copyright  2013-2022.

% PlcByteArray
%>   Represents a PLC array which addresses an unique data entry stored within a programmable logic controller (PLC) its values are interpreted as binary values.
classdef PlcByteArray < PlcArray
    % ---------- Public properties ----------
    properties (Access = public, Dependent = true)
        %> @brief Gets or sets the value assigned to the PLC value.
        %>   The value assigned to the PLC value.
        Value;
    end % properties

    % ---------- Public constructors ----------
    methods (Access = public)
        %> @brief Initializes a new instance of the PlcByteArray class using the specified @b address and @b valuesOrLength.
        %>
        %> @param address The PlcAddress of the new PlcByteArray to which the
        %>                PLC values belong.
        %> @param valuesOrLength
        %>        * @b values: The initial values of the new PlcByteArray.
        %>        * @b length: The array length used to store the array values of the PlcByteArray.
        %>
        %> @b Errors
        %> * The @b address does not refer to the @b PlcRawType.Byte.
        %> * The @b length is out of the bounds defined by @b PlcType.MinLength
        %>   and @b PlcType.MaxLength.
        function this = PlcByteArray(address, valuesOrLength)
            if (iscell(valuesOrLength))
                values = NET.createArray('System.Byte', length(valuesOrLength));

                for index = 1:length(valuesOrLength)
                    values(index) = System.Convert.ToByte(valuesOrLength{index});
                end
            elseif (length(valuesOrLength) > 1)
                values = NET.createArray('System.Byte', length(valuesOrLength));

                for index = 1:length(valuesOrLength)
                   values(index) = System.Convert.ToByte(valuesOrLength(index));
                end
            else
                values = NET.createArray('System.Byte', valuesOrLength);
            end

            this = this@PlcArray( ...
                    PlcTypeCode.Byte, ...
                    address, ...
                    values);
        end
    end % methods
    
    % ---------- Property accessors ----------
    methods
        %> Gets the value assigned to the PLC value.
        function value = get.Value(this)
            value = zeros(1, this.Length, 'uint8');
            managedValue = this.ManagedInstance.GetValue('Value');

            for index = 1:this.Length
                value(index) = managedValue(index);
            end
        end

        %> Sets the value assigned to the PLC value.
        function set.Value(this, value)
            managedValue = this.ManagedInstance.GetValue('Value');

            if (iscell(value))
                for index = 1:min(length(value), this.Length)
                    managedValue.Set(index - 1, System.Convert.ToByte(value{index}));
                end
            elseif (length(value) > 1)
                for index = 1:min(length(value), this.Length)
                    managedValue.Set(index - 1, System.Convert.ToByte(value(index)));
                end
            end

            this.ManagedInstance.SetValue('Value', managedValue);
        end
    end
end

