%> @file PlcDate.m
% This file is protected by TIS
% Copyright  2013-2022.

% PlcDate
%>   Represents a date number value (in PLC a DATE) its value defines an instant in time, where only the date of day is expressed.
classdef PlcDate < PlcValue
    % ---------- Public properties ----------
    properties (Access = public, Dependent = true)
        %> @brief Gets or sets the value assigned to the PLC value.
        %>   The value assigned to the PLC value.
        Value;
    end % properties

    % ---------- Public constructors ----------
    methods (Access = public)
        %> @brief Initializes a new instance of the PlcDate class using the specified @b address and @b value.
        %>
        %> @param address The PlcAddress of the new PlcDate to which the PLC
        %>                value belongs.
        %> @param value The initial value of the new PlcDate.
        %>
        %> @b Errors
        %> * The @b address does not refer to the @b PlcRawType.Word.
        function this = PlcDate(address, value)
            superArguments{1} = PlcTypeCode.Date;
            superArguments{2} = address;
            superArguments{3} = 0; % Length

            if (nargin == 2)
                superArguments{4} = System.DateTime.Parse(datestr(value));
            else
                superArguments{4} = System.DateTime.Now;
            end

            this = this@PlcValue( ...
                    superArguments{1}, ...
                    superArguments{2}, ...
                    superArguments{3}, ...
                    superArguments{4});
        end
    end % methods
    
    % ---------- Property accessors ----------
    methods
        %> Gets the value assigned to the PLC value.
        function value = get.Value(this)
            managedValue = this.ManagedInstance.GetValue('Value');

            value = datenum( ...
                    double(managedValue.Year), ...
                    double(managedValue.Month), ...
                    double(managedValue.Day), ...
                    double(0), ...
                    double(0), ...
                    double(0));
        end

        %> Sets the value assigned to the PLC value.
        function set.Value(this, value)
            this.ManagedInstance.SetValue('Value', System.DateTime.Parse(datestr(value)));
        end
    end % methods
end

