%> @file PlcDevice.m
% This file is protected by TIS
% Copyright  2013-2022.

% PlcDevice
%>   Provides an abstract base class which represents a single programmable logic controller (PLC).
classdef (Abstract = true) PlcDevice < handle
    % ---------- Public properties ----------
    properties (GetAccess = public, SetAccess = private)
        %> @brief Gets or sets the PlcDeviceEndPoint of the PLC device.
        %>   An instance of the PlcDeviceEndPoint class, which defines the
        %>   end point information required to establish a connection to the
        %>   device represented.
        %>
        %> @b Errors
        %> * The @b endPoint is not an instance of the IPDeviceEndPoint class.
        EndPoint;
    end % properties

    % ---------- Internal properties ----------
    properties (GetAccess = public, SetAccess = protected, Hidden = true)
        %> This property is not intended to be used outside of the toolbox.
        Connection = 0;

        %> This property is not intended to be used outside of the toolbox.
        ManagedInstance = 0;
    end % properties

    % ---------- Protected constructors ----------
    methods (Access = protected)
        %> @brief Initializes a new instance of the PlcDevice class using the specified @b endPoint.
        %>
        %> @param endPoint The PlcDeviceEndPoint which identifies the network
        %>                 address of the device.
        %>
        %> @b Errors
        %> * The @b endPoint is not an instance of the IPDeviceEndPoint class.
        function this = PlcDevice(endPoint)
            SimaticMatLabToolbox.Start();
            this.EndPoint = endPoint;
        end
    end % methods

    % ---------- Property accessors ----------
    methods
        %> Gets the PlcDeviceEndPoint of the PLC device.
        function value = get.EndPoint(this)
            value = this.EndPoint;
        end

        %> Sets the PlcDeviceEndPoint of the PLC device.
        function set.EndPoint(this, value)
            if (~isa(value, 'IPDeviceEndPoint'))
                error('The endPoint needs to be an instance of the IPDeviceEndPoint class.');
            end

            this.EndPoint = value;
            this.HandleEndPointChanged();
        end
    end % methods

    % ---------- Public methods ----------
    methods (Access = public, Abstract = true)
        %> @brief Creates a new instance of an object implementing the PlcDeviceConnection class which is associated with the device, if <paramref name="createNew"/> is equals to the value true.
        %>
        %> @param createNew A value indicating whether a new instance is to be created (the value
        %>        true) or if there is already an existing instance of the PlcDeviceConnection
        %>        class in usable state is to be returned (the value false).
        %>
        %> @retval An instance of the PlcDeviceConnection class which is associated with the device.
        %>
        %> @b Remarks
        %>
        %> Any changes made on @b EndPoint and any changes made on the configuration of the
        %> returned @b PlcDeviceConnection will take affect when calling
        %> @b PlcDeviceConnection.Open. Any further changes made on the @b EndPoint or on the
        %> @b PlcDeviceConnection instance will when only take affect after closing (using
        %> @b PlcDeviceConnection.Close) and re-opening (using PlcDeviceConnection.Open) the
        %> connection.
        %>
        %> In multi-threaded environments it is important to take care about which thread does
        %> open/close a connection and which threads do use the same connection concurrent.
        %> Scenarios there multiple threads do share the same connection instance have to ensure
        %> that changes to the @b PlcDevice (e.g. @b EndPoint changes) to that a
        %> @b PlcDeviceConnection does belong or any changes made on the connection itself will
        %> take affect in all threads (which share the same connection instance) immediately after
        %> closing and re-opening the connection.
        %>
        %> If there was a connection created for the first time using this @b PlcDevice instance
        %> these connection will be always returned by any subsequent calls to
        %> @b CreateConnection(bool) using @b createNew with the value false. This is the case
        %> until the cached connection will be discarded. A cached connection will be discarded
        %> then it is not in created nor in closed state and any changes are made to the
        %> configuration of this PlcDevice (e.g. @b EndPoint).
        connection = CreateConnection(createNew);
    end % methods

    % ---------- Private methods ----------
    methods (Access = private)
        %> @brief Handles any changes made to and on the local endPoint variable.
        function HandleEndPointChanged(this)
            %> In case of any changes made to and on the local endPoint variable, reset the local
            %> cached connection to ensure that any call made on CreateConnection using createNew
            %> equals to false does create a new connection using the latest end point information.
            %> Instead of returning the cached connection which has been created using the out
            %> dated end point information.
            %>
            %> Therefore end point changes will only take affect during opening the connnection
            %> there is no need to discard the cached connection as long as it has just created or
            %> it is currently in closed state.
            if (this.Connection ~= 0)
                if (this.Connection.State ~= PlcDeviceConnectionState.Created ...
                        || this.Connection.State ~= PlcDeviceConnectionSate.Closed)
                    this.Connection = 0;
                end
            end
        end
    end % methods
end

