%> @file PlcOperand.m
% This file is protected by TIS
% Copyright  2013-2022.

% PlcOperand
%>   Represents an operand which addresses a specific memory block within a programmable logic controller (PLC).
classdef PlcOperand < handle
    % ---------- Public constants ----------
    properties (Access = public, Constant = true)
        %> @brief The operand used to access the counter memory block of the controller.
        Counter = PlcOperand(PlcOperandType.Counter);

        %> @brief The operand used to access the flag memory block of the controller.
        Flag = PlcOperand(PlcOperandType.Flag);

        %> @brief The operand used to access the input memory block of the controller.
        Input = PlcOperand(PlcOperandType.Input);

        %> @brief The operand used to access the output memory block of the controller.
        Output = PlcOperand(PlcOperandType.Output);

        %> @brief The operand used to access the periphery input memory block of the controller.
        PeripheryInput = PlcOperand(PlcOperandType.PeripheryInput);

        %> @brief The operand used to access the periphery output memory block of the controller.
        PeripheryOutput = PlcOperand(PlcOperandType.PeripheryOutput);

        %> @brief The operand used to access the time memory block of the controller.
        Timer = PlcOperand(PlcOperandType.Timer);
    end % properties
    
    % ---------- Public properties ----------
    properties (Access = public, Dependent = true)
        %> @brief Gets the number associated with the operand.
        %>   The value -1 if the @b PlcOperand.Type is unequals to
        %>   @b PlcOperandType.DataBlock; otherwise a number within the range
        %>   defined by @b MinNumber and @b MaxNumber.
        Number;

        %> @brief Gets the type of operand represented.
        %>   One of the members defined by the PlcOperandType enumeration.
        Type;
    end % properties

    % ---------- Internal properties ----------
    properties (GetAccess = public, SetAccess = protected, Hidden = true)
        %> This property is not intended to be used outside of the toolbox.
        ManagedInstance = 0;
    end % properties

    % ---------- Private constructors ----------
    methods (Access = private)
        %> @brief Initializes a new instance of the PlcOperand class using the @b type and @b number specified.
        %>
        %> @param type The PlcOperandType of the operand to represent.
        %> @param number The number of the data block to represent.
        %>
        %> @b Errors
        %> * The @b PlcOperandType.DataBlock can not be used to create a
        %>   PlcOperand, because it requires a number.
        %> * The number is out of the bounds defined by @b MinNumber or
        %>   @b MaxNumber.
        function this = PlcOperand(type, number)
            SimaticMatLabToolbox.Start();

            if (nargin == 1)
                this.ManagedInstance = IPS7Lnk.Advanced.MatLab.Factory.CreateOperand(int32(type));
            elseif (nargin == 2)
                this.ManagedInstance = IPS7Lnk.Advanced.MatLab.Factory.CreateOperandDataBlock(number);
            end
        end
    end % methods

    % ---------- Property accessors ----------
    methods
        %> Gets the number associated with the operand.
        function value = get.Number(this)
            value = this.ManagedInstance.GetValue('Number');
        end

        %> Gets the type of operand represented.
        function value = get.Type(this)
            value = PlcOperandType(int32(this.ManagedInstance.GetValue('Type')));
        end
    end % methods

    % ---------- Public methods ----------
    methods (Access = public, Static = true)
        %> @brief Retrieves the according PlcOperand of the type @b PlcOperandType.DataBlock using the specified @b number of the data block operand.
        %>
        %> @param number The number of the data block to retrieve.
        %>
        %> @retval value The PlcOperand of a specific data block.
        %>
        %> @b Errors
        %> * The number is out of the bounds defined by @b MinNumber or
        %>   @b MaxNumber.
        function value = DataBlock(number)
            value = PlcOperand(PlcOperandType.DataBlock, number);
        end

        %> @brief Retrieves the according PlcOperand which matches the PlcOperandType specified.
        %>
        %> @param type The PlcOperandType of the PlcOperand to retrieve.
        %>
        %> @retval value The PlcOperand which matches the type.
        %>
        %> @b Errors
        %> * The type is either a @b PlcOperandType.DataBlock or is an unknown
        %>   PlcOperandType.
        function value = Of(type)
            value = PlcOperand(type);
        end
    end % methods
end

