%> @file PlcString.m
% This file is protected by TIS
% Copyright  2013-2022.

% PlcString
%>   Represents a single value (in PLC a TEXT) its value defines a single-precision floating-point number.
classdef PlcString < PlcValue    
    % ---------- Public properties ----------
    properties (Access = public, Dependent = true)
        %> @brief Gets or sets the value assigned to the PLC value.
        %>   The value assigned to the PLC value.
        Value;
    end % properties

    % ---------- Public constructors ----------
    methods (Access = public)
        %> @brief Initializes a new instance of the PlcString class using the specified @b address and @b value.
        %>
        %> @param address The PlcAddress of the new PlcString to which the PLC
        %>                value belongs.
        %> @param value The initial value of the new PlcString.
        %> @param valueLength The maximum length used to store the PlcString
        %>                    value.
        %>
        %> @b Errors
        %> * The @b address does not refer to the @b PlcRawType.Byte.
        function this = PlcString(address, value, valueLength)
            superArguments{1} = PlcTypeCode.String;
            superArguments{2} = address;

            if (ischar(value))
                superArguments{3} = length(value);
                superArguments{4} = value;
            else
                superArguments{3} = int32(value);
                superArguments{4} = System.Convert.ToString('');
            end

            if (nargin == 3)
                superArguments{4} = valueLength;
            else
                superArguments{4} = System.Convert.ToString(value);
            end
            
            this = this@PlcValue( ...
                    superArguments{1}, ...
                    superArguments{2}, ...
                    superArguments{3}, ...
                    superArguments{4});
        end
    end % methods
    
    % ---------- Property accessors ----------
    methods
        %> Gets the value assigned to the PLC value.
        function value = get.Value(this)
            value = char(this.ManagedInstance.GetValue('Value'));
        end
        
        %> Sets the value assigned to the PLC value.
        function set.Value(this, value)
            this.ManagedInstance.SetValue('Value', System.Convert.ToString(value));
        end
    end % methods
end

