%> @file PlcAddress.m
% This file is protected by TIS
% Copyright  2013-2022.

% PlcAddress
%>   Represents an address which refers to a specific data area stored within a programmable logic controller (PLC).
classdef PlcAddress < handle
    % ---------- Public properties ----------
    properties (Access = public, Dependent = true)
        %> @brief Gets the bit number part of the address, which defines to which bit the address refers.
        %>   The number indicating to which specific bit the address refers
        %>   or -1 if PlcRawType is not equals to @b PlcRawType.Bit.
        BitNumber;

        %> @brief Gets the byte number part of the address, which defines to which byte the address refers.
        %>   The number indicating to which specific byte the address
        %>   refers.
        ByteNumber;

        %> @brief Gets the operand number part of the address, which defines to which operand the address refers.
        %>   The number indicating to which specific operand the address
        %>   refers or -1 if PlcOperandType is not equals
        %>   @b PlcOperandType.DataBlock.
        OperandNumber;

        %> @brief Gets the type of operand to which the address refers.
        %>   One of the members defined by the PlcOperandType enumeration.
        %>   Which specifies the type of operand to which the address
        %>   refers.
        OperandType;

        %> @brief Gets the raw type of to which the address refers.
        %>   One of the members defined by the PlcRawType enumeration. Which
        %>   specifies the type to which the address refers.
        RawType;

        %> @brief Gets the string representation of the address.
        %>   A string that contains the address.
        Value;
    end % properties

    % ---------- Internal properties ----------
    properties (GetAccess = public, SetAccess = private, Hidden = true)
        ManagedInstance = 0;
    end % properties

    % ---------- Public constructors ----------
    methods (Access = public)
        %> @brief Initializes a new instance of the PlcAddress class using the specified @b operand, @b rawType and @b byteNumber.
        %>
        %> @param operand The PlcOperand to that the new PlcAddress refers.
        %> @param rawType The PlcRawType to that the new PlcAddress refers.
        %> @param byteNumber The number of the byte to that the new PlcAddress
        %>                   refers.
        %> @param bitNumber The number of the bit to that the new PlcAddress
        %>                  refers.
        %>
        %> @b Errors
        %> * Specifing a @b rawType equals to @b PlcRawType.Bit requires a bit
        %>   number.
        %> * The @b byteNumber or @b bitNumber is out of the bounds defined by
        %>   @b MinByteNumber, @b MaxByteNumber, @b MinBitNumber or @b MaxBitNumber.
        function this = PlcAddress(operand, rawType, byteNumber, bitNumber)
            SimaticMatLabToolbox.Start();

            if (nargin == 3)
                this.ManagedInstance = IPS7Lnk.Advanced.MatLab.Factory.CreateAddress( ...
                        operand.ManagedInstance, ...
                        int32(rawType), ...
                        byteNumber);
            elseif (nargin == 4)
                this.ManagedInstance = IPS7Lnk.Advanced.MatLab.Factory.CreateAddress( ...
                        operand.ManagedInstance, ...
                        int32(rawType), ...
                        byteNumber, ...
                        bitNumber);
            end
        end
    end % methods
    
    % ---------- Property accessors ----------
    methods
        %> Gets the bit number part of the address, which defines to which bit the address refers.
        function value = get.BitNumber(this)
            value = this.ManagedInstance.GetValue('BitNumber');
        end

        %> Gets the byte number part of the address, which defines to which byte the address refers.
        function value = get.ByteNumber(this)
            value = this.ManagedInstance.GetValue('ByteNumber');
        end

        %> Gets the operand number part of the address, which defines to which operand the address refers.
        function value = get.OperandNumber(this)
            value = this.ManagedInstance.GetValue('OperandNumber');
        end

        %> Gets the type of operand to which the address refers.
        function value = get.OperandType(this)
            value = PlcOperandType(int32(this.ManagedInstance.GetValue('OperandType')));
        end

        %> Gets the raw type of to which the address refers.
        function value = get.RawType(this)
            value = PlcRawType(int32(this.ManagedInstance.GetValue('RawType')));
        end

        %> Gets the string representation of the address.
        function value = get.Value(this)
            value = char(this.ManagedInstance.Invoke('ToString'));
        end
    end % methods
end

