%> @file PlcBoolean.m
% This file is protected by TIS
% Copyright  2013-2022.

% PlcBoolean
%>   Represents a logical value (in PLC a BOOL) its value is interpreted as a logical expression.
classdef PlcBoolean < PlcValue    
    % ---------- Public properties ----------
    properties (Access = public, Dependent = true)
        %> @brief Gets or sets the value assigned to the PLC value.
        %>   The value assigned to the PLC value.
        Value;
    end % properties

    % ---------- Public constructors ----------
    methods (Access = public)
        %> @brief Initializes a new instance of the PlcBoolean class using the specified address and value.
        %>
        %> @param address The PlcAddress of the new PlcBoolean to which the PLC
        %>                value belongs.
        %> @param value The initial value of the new PlcBoolean.
        %>
        %> @b Errors
        %> * The address does not refer to the @b PlcRawType.Bit.
        function this = PlcBoolean(address, value)
            superArguments{1} = PlcTypeCode.Bool;
            superArguments{2} = address;
            superArguments{3} = 0; % Length

            if (nargin == 2)
                superArguments{4} = System.Convert.ToBoolean(value);
            else
                superArguments{4} = System.Convert.ToBoolean(false);
            end

            this = this@PlcValue( ...
                    superArguments{1}, ...
                    superArguments{2}, ...
                    superArguments{3}, ...
                    superArguments{4});
        end
    end % methods

    % ---------- Property accessors ----------
    methods
        %> Gets the value assigned to the PLC value.
        function value = get.Value(this)
            value = this.ManagedInstance.GetValue('Value');
        end

        %> Sets the value assigned to the PLC value.
        function set.Value(this, value)
            this.ManagedInstance.SetValue('Value', System.Convert.ToBoolean(value));
        end
    end % methods
end

