%> @file PlcDateTime.m
% This file is protected by TIS
% Copyright  2013-2022.

% PlcDateTime
%>   Represents a date number value (in PLC a DATE_AND_TIME) its value defines an instant in time, typically expressed as a date and time of day.
classdef PlcDateTime < PlcValue
    % ---------- Public properties ----------
    properties (Access = public, Dependent = true)
        %> @brief Gets or sets the value assigned to the PLC value.
        %>   The value assigned to the PLC value.
        Value;
    end % properties

    % ---------- Public constructors ----------
    methods (Access = public)
        %> @brief Initializes a new instance of the PlcDateTime class using the specified address and value.
        %>
        %> @param address The PlcAddress of the new PlcDateTime to which the PLC
        %>                value belongs.
        %> @param value The initial @b value of the new PlcDateTime.
        %>
        %> @b Errors
        %> * The @b address does not refer to the @b PlcRawType.Byte.
        function this = PlcDateTime(address, value)
            superArguments{1} = PlcTypeCode.DateTime;
            superArguments{2} = address;
            superArguments{3} = 0; % Length

            if (nargin == 2)
                superArguments{4} = System.DateTime.Parse(datestr(value));
            else
                superArguments{4} = System.DateTime.Now;
            end

            this = this@PlcValue( ...
                    superArguments{1}, ...
                    superArguments{2}, ...
                    superArguments{3}, ...
                    superArguments{4});
        end
    end % methods

    % ---------- Property accessors ----------
    methods
        %> Gets the value assigned to the PLC value.
        function value = get.Value(this)
            managedValue = this.ManagedInstance.GetValue('Value');

            value = datenum( ...
                    double(managedValue.Year), ...
                    double(managedValue.Month), ...
                    double(managedValue.Day), ...
                    double(managedValue.Hour), ...
                    double(managedValue.Minute), ...
                    double(managedValue.Second));
        end

        %> Sets the value assigned to the PLC value.
        function set.Value(this, value)
            this.ManagedInstance.SetValue('Value', System.DateTime.Parse(datestr(value)));
        end
    end % methods
end

