%> @file PlcS5Time.m
% This file is protected by TIS
% Copyright  2013-2022.

% PlcS5Time
%>   Represents a date number value, where only the time of day is expressed, (in PLC a S5TIME) its value defines an time interval.
classdef PlcS5Time < PlcValue
    % ---------- Public properties ----------
    properties (Access = public, Dependent = true)
        %> @brief Gets or sets the value assigned to the PLC value.
        %>   The value assigned to the PLC value.
        Value;
    end % properties

    % ---------- Public constructors ----------
    methods (Access = public)
        %> @brief Initializes a new instance of the PlcS5Time class using the specified @b address and @b value.
        %>
        %> @param address The PlcAddress of the new PlcS5Time to which the
        %>                PLC value belongs.
        %> @param value The initial value of the new PlcS5Time.
        %>
        %> @b Errors
        %> * The @b address does not refer to the @b PlcRawType.Word.
        function this = PlcS5Time(address, value)
            superArguments{1} = PlcTypeCode.S5Time;
            superArguments{2} = address;
            superArguments{3} = 0; % Length

            if (nargin == 2)
                superArguments{4} = System.DateTime.Parse(datestr(value)).TimeOfDay;
            else
                superArguments{4} = System.DateTime.Now.TimeOfDay;
            end

            this = this@PlcValue( ...
                    superArguments{1}, ...
                    superArguments{2}, ...
                    superArguments{3}, ...
                    superArguments{4});
        end
    end % methods

    % ---------- Property accessors ----------
    methods
        %> Gets the value assigned to the PLC value.
        function value = get.Value(this)
            managedValue = this.ManagedInstance.GetValue('Value');

            value = datenum( ...
                    double(0), ...
                    double(0), ...
                    double(0), ...
                    double(managedValue.Hours), ...
                    double(managedValue.Minutes), ...
                    double(managedValue.Seconds));
        end

        %> Sets the value assigned to the PLC value.
        function set.Value(this, value)
            this.ManagedInstance.SetValue('Value', System.DateTime.Parse(datestr(value)).TimeOfDay);
        end
    end % methods
end

