%> @file PlcValue.m
% This file is protected by TIS
% Copyright  2013-2022.

% PlcValue
%>   Represents an abstract base class of PLC values.
classdef (Abstract = true) PlcValue < handle
    % ---------- Public properties ----------
    properties (GetAccess = public, Dependent = true)
        %> @brief Gets the address of the PLC value.
        %>   The <see cref="PlcAddress" /> used by the PLC value.
        Address;
    end % properties

    properties (GetAccess = public, Dependent = true, Abstract = true)
        %> @brief Gets or sets the value assigned to the PLC value.
        %>   The value assigned to the PLC value.
        Value;
    end % properties
    
    % ---------- Internal properties ----------
    properties (GetAccess = public, SetAccess = protected, Hidden = true)
        %> This property is not intended to be used outside of the toolbox.
        ManagedInstance = 0;
    end % properties

    % ---------- Protected constructors ----------
    methods (Access = protected)
        %> @brief Initializes a new instance of the PlcValue class.
        %>
        %> @param typeCode One of the members defined by the PlcTypeCode
        %>                 enumeration class which defines the concrete type of
        %>                 value represented.
        %> @param address The PlcAddress of the new PlcValue to which the PLC
        %>                value belongs.
        %> @param length The length of data represented.
        %> @param value The initial value of the new PlcValue.
        function this = PlcValue(typeCode, address, length, value)
            SimaticMatLabToolbox.Start();
            
            if (isa(address, 'PlcAddress'))
                address = address.Value;
            end

            if (length == 0)
                type = IPS7Lnk.Advanced.MatLab.Factory.CreateType( ...
                        int32(typeCode), address);
            else
                type = IPS7Lnk.Advanced.MatLab.Factory.CreateType( ...
                        int32(typeCode), address, length);
            end

            this.ManagedInstance = IPS7Lnk.Advanced.MatLab.Factory.CreateValue(type, value);
        end
    end % methods

    % ---------- Property accessors ----------
    methods
        %> Gets the address of the PLC value.
        function value = get.Address(this)
            value = char(this.ManagedInstance.GetValue('AddressValue'));
        end
    end % methods

    % ---------- Public methods ----------
    methods (Access = public)
        %> @brief Retrieves the current value of the PLC value from the source specified.
        %>
        %> @param this The instance of the PlcValue class to operate on.
        %> @param source The PlcDevice or PlcDeviceConnection from that the data
        %>               is to be retrieved.
        %>
        %> @retval value The current value of the PLC value.
        function value = GetValue(this, source)
            value = IPS7Lnk.Advanced.MatLab.Helper.GetValue( ...
                    source.ManagedInstance, ...
                    this.ManagedInstance);
        end
        
        %> @brief Determines whether a specified value is acceptable for this PlcValue.
        %>
        %> @param this The instance of the PlcValue class to operate on.
        %> @param value The value to check.
        %>
        %> @retval value The value true, if the specified value is the type or an
        %>               acceptable derived type; otherwise the value false.
        function value = IsValidType(this, value)
            value = this.ManagedInstance.Invoke('IsValidType', value);
        end
        
        %> @brief Determines whether the provided value is accepted for the type of PLC value through basic type checking and also potentially if it is within the allowed range of value for that type.
        %>
        %> @param this The instance of the PlcValue class to operate on.
        %> @param value The value to check.
        %>
        %> @retval value The value true, if the specified value is acceptable and is of
        %>               the correct type or a derived type; otherwise the value false.
        function value = IsValidValue(this, value)
            value = this.ManagedInstance.Invoke('IsValidValue', value);
        end

        %> @brief Stores the value in the target specified.
        %>
        %> @param this The instance of the PlcValue class to operate on.
        %> @param target The PlcDevice or PlcDeviceConnection to use to store
        %>               the data.
        %> @param value The value to be stored.
        %>
        %> @b Errors
        %> * The @b value is invalid. It either does not fall within an
        %>   expected value range or its type is not supported by the
        %>   PlcValue.
        function SetValue(this, target, value)
            IPS7Lnk.Advanced.MatLab.Helper.SetValue( ...
                    target.ManagedInstance, ...
                    this.ManagedInstance, ...
                    value);
        end
    end % methods
end

