﻿// Copyright (c) Traeger Industry Components GmbH.  All Rights Reserved.

namespace CS_Server_Graphical
{
    using System;
    using System.ComponentModel;
    using System.Net;
    using System.Text;
    using System.Windows;
    using System.Windows.Controls;
    using System.Windows.Documents;

    using Rfc1006LibNet.Advanced;
    using Rfc1006LibNet.Advanced.EventArgs;

    public partial class TransferWindow : Window
    {
        private Rfc1006Server server;

        public TransferWindow(Rfc1006Server server)
        {
            InitializeComponent();

            this.server = server;

            #region Connection Information

            this.tbIpAddress.Text = (server.EndPoint.Address != IPAddress.None) ? server.EndPoint.Address.ToString() : string.Empty;
            this.tbPort.Text = server.EndPoint.Port.ToString();
            this.tbLocalTsap.Text = Encoding.Default.GetString(server.EndPoint.LocalTSAP);
            this.tbRemoteTsap.Text = Encoding.Default.GetString(server.EndPoint.RemoteTSAP);

            #endregion

            #region Configuration Information

            this.tbReceiveTiemout.Text = server.ReceiveTimeout.ToString();
            this.tbTransmitTimeout.Text = server.TransmitTimeout.ToString();
            this.tbAliveTime.Text = (this.server.AliveTime >= 0) ? this.server.AliveTime.ToString() : string.Empty;

            #endregion

            #region 1. Receive Option (currently not in use)

            //receiveManuell();

            #endregion

            #region 2. Receive Option (currently in use)

            // mount event after a receive was done
            this.server.AutoReceive = true;
            this.server.Received += this.server_Received;

            #endregion

            // mount event at status changed
            this.server.StatusChanged += this.server_StatusChanged;

            lbStatus.Content = this.server.Status;
        }

        #region 1. Receive Option (currently not in use)

        /*private void receiveManuell()
        {
            new Thread(() =>
                {
                    string oldStatus = string.Empty;

                    while (true)
                    {
                        try
                        {
                            string ret = server.ReceiveString();

                            this.Dispatcher.InvokeAsync(new Action(() =>
                            {
                                this.tbReceived.Text = ret;
                            }), System.Windows.Threading.DispatcherPriority.DataBind);
                        }
                        catch (TimeoutException)
                        {
                        }
                    }
                }).Start();
        }*/

        #endregion

        #region 2. Receive Option (currently in use)

        private void server_Received(object sender, TransferEventArgs e)
        {
            this.Dispatcher.InvokeAsync(new Action(() => {
                this.tbReceived.Text = Encoding.Default.GetString(e.Buffer);

                if (cbEchoSend.IsChecked == true)
                    this.server.Transmit(this.tbReceived.Text);
            }), System.Windows.Threading.DispatcherPriority.DataBind);
        }

        #endregion

        #region Transmit

        private void OnSend(object sender, RoutedEventArgs e)
        {
            this.send();
        }

        private void OnTbSendTextChanged(object sender, TextChangedEventArgs e)
        {
            if (this.cbAutoSend.IsChecked == true)
                this.send();
        }

        private void send()
        {
            try {
                // get the text from the textbox
                string sendTxt = new TextRange(this.tbSend.Document.ContentStart,
                                                this.tbSend.Document.ContentEnd).Text;

                // send the whole text of the textbox to the stream
                this.server.Transmit(sendTxt);
            }
            catch (TimeoutException) {
            }
            catch (Exception ex) {
                MessageBox.Show(ex.Message);
            }
        }

        #endregion

        #region Status & Configuration

        private void server_StatusChanged(object sender, StatusChangedEventArgs e)
        {
            this.Dispatcher.InvokeAsync(new Action(() => {
                this.lbStatus.Content = e.Status;

                if (e.Exception != null && e.Status == Rfc1006Status.ConnectionClosing) {
                    this.tbErrorMessage.Text = e.Exception.GetType() + "\n\n" + e.Exception.Message;

                    this.tbErrorMessage.Visibility = System.Windows.Visibility.Visible;
                }
                else if (e.Status == Rfc1006Status.Connected) {
                    this.ConfigurationInformationIsEnabled(false);
                    this.tbErrorMessage.Visibility = System.Windows.Visibility.Hidden;
                }
            }), System.Windows.Threading.DispatcherPriority.Send);
        }

        private void ConfigurationInformationIsEnabled(bool value)
        {
            this.btStart.IsEnabled = value;
            this.btStop.IsEnabled = !value;

            this.tbSend.IsEnabled = !value;
            this.btSend.IsEnabled = !value;
            this.cbAutoSend.IsEnabled = !value;

            this.tbIpAddress.IsEnabled = value;
            this.tbPort.IsEnabled = value;
            this.tbLocalTsap.IsEnabled = value;
            this.tbRemoteTsap.IsEnabled = value;
            this.tbAliveTime.IsEnabled = value;
        }

        private void OnWindowClosing(object sender, CancelEventArgs e)
        {
            this.server.Stop(); // optional, will be done at disposing if not done by user  
            Environment.Exit(0);
        }

        private void OnConfigurationChanged(object sender, RoutedEventArgs e)
        {
            #region Connection Information

            IPAddress address = IPAddress.None;
            if (IPAddress.TryParse(this.tbIpAddress.Text, out address))
                this.server.EndPoint.Address = IPAddress.Parse(this.tbIpAddress.Text);

            this.server.EndPoint.Port = Int32.Parse(this.tbPort.Text);
            this.server.EndPoint.LocalTSAP = Encoding.Default.GetBytes(this.tbLocalTsap.Text);
            this.server.EndPoint.RemoteTSAP = Encoding.Default.GetBytes(this.tbRemoteTsap.Text);

            #endregion

            #region Configuration Information

            this.server.ReceiveTimeout = Int32.Parse(this.tbReceiveTiemout.Text);
            this.server.TransmitTimeout = Int32.Parse(this.tbTransmitTimeout.Text);
            this.server.AliveTime = (this.tbAliveTime.Text != String.Empty) ? Int32.Parse(this.tbAliveTime.Text) : -1;

            #endregion
        }

        private void OnStart(object sender, RoutedEventArgs e)
        {
            this.ConfigurationInformationIsEnabled(false);

            this.server.Start();
        }

        private void OnStop(object sender, RoutedEventArgs e)
        {
            this.ConfigurationInformationIsEnabled(true);

            this.server.Stop();
        }

        #endregion

        private void OnHelp(object sender, RoutedEventArgs e)
        {
            System.Windows.Forms.Help.ShowHelp(null, StartWindow.HelpPath);
        }

        private void OnInfo(object sender, RoutedEventArgs e)
        {
            System.Windows.Forms.Help.ShowHelp(null, StartWindow.InfoPath);
        }

        private void OnExit(object sender, RoutedEventArgs e)
        {
            this.server.Stop(); // optional, will be done at disposing if not done by user  
            Environment.Exit(0);
        }

        private void OnCleanSend(object sender, RoutedEventArgs e)
        {
            this.tbSend.Document.Blocks.Clear();
        }

        private void OnCleanReceive(object sender, RoutedEventArgs e)
        {
            this.tbReceived.Text = string.Empty;
        }
    }
}
