﻿// Copyright (c) Traeger Industry Components GmbH.  All Rights Reserved.

namespace CS_Client_Events
{
    using System;
    using System.Text;

    using Rfc1006LibNet.Advanced;
    using Rfc1006LibNet.Advanced.EventArgs;

    /// <summary>
    /// This sample demonstrates a Hello World! application.
    /// </summary>
    /// <remarks>
    /// This application does write/read the 'Hello World!' message via RFC1006
    /// and prints the message on the standard output 
    /// from a client perspective with any RFC1006 client.
    /// </remarks>
    public class Program
    {
        public static void Main(string[] args)
        {
            // Connect to any RFC1006 client 
            // using the specified IP and TSAPs and the default values 
            // for the not defined attributes 
            // like port, connection timeout, receive timeout...
            Console.WriteLine("Starting client... \n");
            Rfc1006Client client = new Rfc1006Client("127.0.0.1", "localTSAP", "remoteTSAP");

            Console.WriteLine("###########################################################");

            // Bind on the event before the connection
            client.Connecting += client_Connecting;

            // Bind on the event after the connection
            client.Connected += client_Connected;

            // Bind on the event before the disconnection
            client.Disconnecting += client_Disconnecting;

            // Bind on the event after the disconnection
            client.Disconnected += client_Disconnected;

            // Bind on the status changed event
            client.StatusChanged += client_StatusChanged;

            // Bind on the event before the transmit
            client.Transmitting += client_Transmitting;

            // Bind on the event after the transmit
            client.Transmitted += client_Transmitted;

            // Instruct the client to do a automatic receive
            client.AutoReceive = true;
            client.Received += client_Received;

            client.Start();

            // Transmits the content of the entered line at the console.
            string str = string.Empty;
            while ((str = Console.ReadLine()) != "exit" && str != "quit")
                client.Transmit(str);

            Console.WriteLine("###########################################################");
            Console.WriteLine("Client disconnecting! \n");
            client.Stop();

            Console.WriteLine("###########################################################");
            Console.WriteLine("Press key to exit! \n");
            Console.ReadKey();
        }

        private static void client_Received(object sender, TransferEventArgs e)
        {
            Console.WriteLine("Received done! Data= " + Encoding.Default.GetString(e.Buffer));
        }

        private static void client_Transmitted(object sender, TransferEventArgs e)
        {
            Console.WriteLine("Transfer done! Data= " + Encoding.Default.GetString(e.Buffer));
        }

        private static void client_Transmitting(object sender, TransferEventArgs e)
        {
            Console.WriteLine("Doing transfer... (and add \"Hello World\" to the Packet)");

            string addToBuffer = "Hello World";

            byte[] buffer = e.Buffer;
            Array.Resize<Byte>(ref buffer, e.Buffer.Length + addToBuffer.Length);
            Array.Copy(
                    Encoding.Default.GetBytes(addToBuffer),
                    0,
                    buffer,
                    buffer.Length - addToBuffer.Length,
                    addToBuffer.Length);
            e.Buffer = buffer;
        }

        private static void client_Disconnected(object sender, ConnectionEventArgs e)
        {
            Console.WriteLine("Disconnection done!");
        }

        private static void client_Disconnecting(object sender, ConnectionEventArgs e)
        {
            Console.WriteLine("Doing disconnection...");
        }

        private static void client_Connected(object sender, ConnectionEventArgs e)
        {
            Console.WriteLine("Connection done!");
        }

        private static void client_Connecting(object sender, ConnectionEventArgs e)
        {
            Console.WriteLine("Doing connection...");
        }

        private static void client_StatusChanged(object sender, StatusChangedEventArgs e)
        {
            Console.WriteLine("     ->  New Status: " + e.Status);
        }
    }
}
