﻿// Copyright (c) Traeger Industry Components GmbH.  All Rights Reserved.

namespace CS_Client_Graphical
{
    using System;
    using System.ComponentModel;
    using System.Net;
    using System.Text;
    using System.Windows;
    using System.Windows.Controls;
    using System.Windows.Documents;

    using Rfc1006LibNet.Advanced;
    using Rfc1006LibNet.Advanced.EventArgs;

    public partial class TransferWindow : Window
    {
        private Rfc1006Client client;

        public TransferWindow(Rfc1006Client client)
        {
            InitializeComponent();

            this.client = client;

            #region Connection Information

            this.tbIpAddress.Text = client.EndPoint.Address.ToString();
            this.tbPort.Text = client.EndPoint.Port.ToString();
            this.tbLocalTsap.Text = Encoding.Default.GetString(client.EndPoint.LocalTSAP);
            this.tbRemoteTsap.Text = Encoding.Default.GetString(client.EndPoint.RemoteTSAP);

            #endregion

            #region Configuration Information

            this.tbConnectionTimeout.Text = client.ConnectionTimeout.ToString();
            this.tbReceiveTiemout.Text = client.ReceiveTimeout.ToString();
            this.tbTransmitTimeout.Text = client.TransmitTimeout.ToString();
            this.tbAliveTime.Text = (client.AliveTime >= 0) ? client.AliveTime.ToString() : string.Empty;

            #endregion

            #region 1. Receive Option (currently not in use)

            //receiveManuell();

            #endregion

            #region 2. Receive Option (currently in use)

            // mount event after a receive was done
            this.client.AutoReceive = true;
            this.client.Received += this.client_Received;

            #endregion

            // mount event at status changed
            this.client.StatusChanged += this.client_StatusChanged;

            lbStatus.Content = this.client.Status;
        }

        #region 1. Receive Option (currently not in use)

        /*private void receiveManuell()
        {
            new Thread(() =>
                {
                    string oldStatus = string.Empty;

                    while (true)
                    {
                        try
                        {
                            string ret = client.ReceiveString();

                            this.Dispatcher.InvokeAsync(new Action(() =>
                            {
                                this.tbReceived.Text = ret;
                            }), System.Windows.Threading.DispatcherPriority.DataBind);
                        }
                        catch (TimeoutException)
                        {
                        }
                    }
                }).Start();
        }*/

        #endregion

        #region 2. Receive Option (currently in use)

        private void client_Received(object sender, TransferEventArgs e)
        {
            this.Dispatcher.InvokeAsync(new Action(() => {
                this.tbReceived.Text = Encoding.Default.GetString(e.Buffer);
            }), System.Windows.Threading.DispatcherPriority.DataBind);
        }

        #endregion

        #region Transmit

        private void OnSend(object sender, RoutedEventArgs e)
        {
            this.send();
        }

        private void OnTbSendTextChanged(object sender, TextChangedEventArgs e)
        {
            if (this.cbAutoSend.IsChecked == true)
                this.send();
        }

        private void send()
        {
            try {
                // get the text from the textbox
                string sendTxt = new TextRange(this.tbSend.Document.ContentStart,
                                                this.tbSend.Document.ContentEnd).Text;

                // send the whole text of the textbox to the stream
                this.client.Transmit(sendTxt);
            }
            catch (TimeoutException) {
            }
            catch (Exception ex) {
                MessageBox.Show(ex.Message);
            }
        }

        #endregion

        #region Status & Configuration

        private void client_StatusChanged(object sender, StatusChangedEventArgs e)
        {
            this.Dispatcher.InvokeAsync(new Action(() => {
                this.lbStatus.Content = e.Status;

                if (e.Exception != null && e.Status == Rfc1006Status.ConnectionClosing) {
                    this.tbErrorMessage.Text = e.Exception.GetType() + "\n\n" + e.Exception.Message;

                    if (this.cbAutomaticReconnection.IsChecked == false) {
                        this.ConfigurationInformationIsEnabled(true);
                    }

                    this.tbErrorMessage.Visibility = System.Windows.Visibility.Visible;
                }
                else if (e.Status == Rfc1006Status.Connected) {
                    this.ConfigurationInformationIsEnabled(false);
                    this.tbErrorMessage.Visibility = System.Windows.Visibility.Hidden;
                }

                this.cbAutomaticReconnection.IsChecked = client.AutoReconnect;
            }), System.Windows.Threading.DispatcherPriority.Send);
        }

        private void ConfigurationInformationIsEnabled(bool value)
        {
            if (this.cbAutomaticReconnection.IsChecked == false)
                this.btConnect.IsEnabled = value;
            else
                this.btConnect.IsEnabled = false;

            this.btDisconnect.IsEnabled = !value;

            this.tbSend.IsEnabled = !value;
            this.btSend.IsEnabled = !value;
            this.cbAutoSend.IsEnabled = !value;

            this.tbIpAddress.IsEnabled = value;
            this.tbPort.IsEnabled = value;
            this.tbLocalTsap.IsEnabled = value;
            this.tbRemoteTsap.IsEnabled = value;
            this.tbConnectionTimeout.IsEnabled = value;
            this.tbAliveTime.IsEnabled = value;
        }

        private void OnWindowClosing(object sender, CancelEventArgs e)
        {
            this.client.Stop(); // optional, will be done at disposing if not done by user  
            Environment.Exit(0);
        }

        private void OnConfigurationChanged(object sender, RoutedEventArgs e)
        {
            #region Connection Information

            this.client.AutoReconnect = (bool)this.cbAutomaticReconnection.IsChecked;

            this.client.EndPoint.Address = IPAddress.Parse(this.tbIpAddress.Text);
            this.client.EndPoint.Port = Int32.Parse(this.tbPort.Text);
            this.client.EndPoint.LocalTSAP = Encoding.Default.GetBytes(this.tbLocalTsap.Text);
            this.client.EndPoint.RemoteTSAP = Encoding.Default.GetBytes(this.tbRemoteTsap.Text);

            #endregion

            #region Configuration Information

            this.client.ConnectionTimeout = Int32.Parse(this.tbConnectionTimeout.Text);
            this.client.ReceiveTimeout = Int32.Parse(this.tbReceiveTiemout.Text);
            this.client.TransmitTimeout = Int32.Parse(this.tbTransmitTimeout.Text);
            this.client.AliveTime = (this.tbAliveTime.Text != String.Empty) ? Int32.Parse(this.tbAliveTime.Text) : -1;

            #endregion
        }

        private void OnConnect(object sender, RoutedEventArgs e)
        {
            this.ConfigurationInformationIsEnabled(false);
            this.client.Start();
        }

        private void OnDisconnect(object sender, RoutedEventArgs e)
        {
            this.cbAutomaticReconnection.IsChecked = false;
            this.ConfigurationInformationIsEnabled(true);
            this.client.Stop();
        }

        #endregion

        private void OnHelp(object sender, RoutedEventArgs e)
        {
            System.Windows.Forms.Help.ShowHelp(null, LoginWindow.HelpPath);
        }

        private void OnInfo(object sender, RoutedEventArgs e)
        {
            System.Windows.Forms.Help.ShowHelp(null, LoginWindow.InfoPath);
        }

        private void OnExit(object sender, RoutedEventArgs e)
        {
            this.client.Stop(); // optional, will be done at disposing if not done by user  
            Environment.Exit(0);
        }

        private void OnCleanSend(object sender, RoutedEventArgs e)
        {
            this.tbSend.Document.Blocks.Clear();
        }

        private void OnCleanReceive(object sender, RoutedEventArgs e)
        {
            this.tbReceived.Text = string.Empty;
        }
    }
}
