﻿// Copyright (c) Traeger Industry Components GmbH.  All Rights Reserved.

namespace CS_Server_Events
{
    using System;
    using System.Text;

    using Rfc1006LibNet.Advanced;
    using Rfc1006LibNet.Advanced.EventArgs;

    /// <summary>
    /// <summary>
    /// This sample demonstrates a Hello World! application.
    /// </summary>
    /// <remarks>
    /// This application does write/read the 'Hello World!' message via RFC1006
    /// and prints the message on the standard output 
    /// from a server perspective with any RFC1006 server.
    /// </remarks>
    public class Program
    {
        public static void Main(string[] args)
        {
            // Connect to any RFC1006 server 
            // using the specified IP and TSAPs and the default values 
            // for the not defined attributes 
            // like port...
            Console.WriteLine("Starting server... \n");
            Rfc1006Server server = new Rfc1006Server("a", "b");

            // Bind on the event after the connection
            server.Connected += server_Connected;

            // Bind on the event before the start of the server
            server.Starting += server_Starting;

            // Bind on the event after the start of the server
            server.Started += server_Started;

            // Bind on the event before the stop of the server
            server.Stopping += server_Stopping;

            // Bind on the event after the stop of the server
            server.Stopped += server_Stopped;

            // Bind on the status changed event
            server.StatusChanged += server_StatusChanged;

            // Bind on the event before the transmit
            server.Transmitting += server_Transmitting;

            // Bind on the event after the transmit
            server.Transmitted += server_Transmitted;

            // Instruct the server to do a automatic receive
            server.AutoReceive = true;
            server.Received += server_Received;

            server.Start();

            // Wait until a client connected.
            Console.WriteLine("###########################################################");
            Console.WriteLine("Wait until a client connected... \n");
            server.Connect();

            Console.WriteLine("###########################################################");

            // Transmits the content of the entered line at the console.
            string str = string.Empty;
            while ((str = Console.ReadLine()) != "exit" && str != "quit")
                server.Transmit(str);

            Console.WriteLine("###########################################################");
            Console.WriteLine("Success, server stopping! \n");
            server.Stop();

            Console.WriteLine("###########################################################");
            Console.WriteLine("Press key to exit! \n");
            Console.ReadKey();
        }

        static void server_Stopped(object sender, ConnectionEventArgs e)
        {
            Console.WriteLine("Server stopped!");
        }

        static void server_Stopping(object sender, ConnectionEventArgs e)
        {
            Console.WriteLine("Server stopping..");
        }

        static void server_Started(object sender, ConnectionEventArgs e)
        {
            Console.WriteLine("Server started!");
        }

        static void server_Starting(object sender, ConnectionEventArgs e)
        {
            Console.WriteLine("Server starting..");
        }

        private static void server_Received(object sender, TransferEventArgs e)
        {
            Console.WriteLine("Received done! Data= " + Encoding.Default.GetString(e.Buffer));
        }

        private static void server_Transmitted(object sender, TransferEventArgs e)
        {
            Console.WriteLine("Transfer done! Data= " + Encoding.Default.GetString(e.Buffer));
        }

        private static void server_Transmitting(object sender, TransferEventArgs e)
        {
            Console.WriteLine("Doing transfer... (and add \"Hello World\" to the Packet)");

            string addToBuffer = "Hello World";

            byte[] buffer = e.Buffer;
            Array.Resize<Byte>(ref buffer, e.Buffer.Length + addToBuffer.Length);
            Array.Copy(Encoding.Default.GetBytes(addToBuffer), 0, buffer, buffer.Length - addToBuffer.Length, addToBuffer.Length);
            e.Buffer = buffer;
        }

        private static void server_Connected(object sender, ConnectionEventArgs e)
        {
            Console.WriteLine("Connection done!");
        }

        private static void server_StatusChanged(object sender, StatusChangedEventArgs e)
        {
            Console.WriteLine("     ->  New Status: " + e.Status);
        }
    }
}
